<?php
class ProofOfWork {
    private $difficulty = 2;
    private $sessionTimeout = 3600; 
    public function generateChallenge() {
        $challenge = bin2hex(random_bytes(32));
        $_SESSION['pow_challenge'] = $challenge;
        $_SESSION['pow_challenge_time'] = time();
        return $challenge;
    }
    public function verifyProof($challenge, $nonce, $solution) {
        if (!isset($_SESSION['pow_challenge'])) {
            return false;
        }
        
        if ($_SESSION['pow_challenge'] !== $challenge) {
            return false;
        }
        if (time() - $_SESSION['pow_challenge_time'] > $this->sessionTimeout) {
            return false;
        }
        $hash = hash('sha256', $challenge . $nonce);
        if (!$this->checkDifficulty($hash)) {
            return false;
        }
        $_SESSION['pow_verified'] = true;
        $_SESSION['pow_verified_time'] = time();
        unset($_SESSION['pow_challenge']);
        return true;
    }
    private function checkDifficulty($hash) {
        $leadingZeros = 0;
        for ($i = 0; $i < strlen($hash); $i++) {
            if ($hash[$i] === '0') {
                $leadingZeros++;
            } else {
                break;
            }
        }
        return $leadingZeros >= $this->difficulty;
    }
    public function isVerified() {
        if (!isset($_SESSION['pow_verified'])) {
            return false;
        }
        
        // Cek apakah token masih valid (24 jam)
        if (time() - $_SESSION['pow_verified_time'] > 86400) {
            unset($_SESSION['pow_verified']);
            return false;
        }
        
        return true;
    }
    public function getDifficulty() {
        return $this->difficulty;
    }
}
$pow = new ProofOfWork();
$requestMethod = $_SERVER['REQUEST_METHOD'];
$contentType = isset($_SERVER['CONTENT_TYPE']) ? $_SERVER['CONTENT_TYPE'] : '';
if ($requestMethod === 'GET' && isset($_GET['get_challenge'])) {
    $challenge = $pow->generateChallenge();
    $difficulty = $pow->getDifficulty();
    
    echo json_encode([
        'challenge' => $challenge,
        'difficulty' => $difficulty
    ]);
    exit;
}
if ($requestMethod === 'POST' && isset($_GET['verify_proof'])) {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($input['challenge'], $input['nonce'], $input['solution'])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Parameter tidak lengkap']);
        exit;
    }
    
    $verified = $pow->verifyProof(
        $input['challenge'],
        $input['nonce'],
        $input['solution']
    );
    
    if ($verified) {
        echo json_encode(['success' => true, 'message' => 'Verifikasi berhasil']);
    } else {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Proof of Work tidak valid']);
    }
    exit;
}
if ($requestMethod === 'GET' && isset($_GET['check_verified'])) {
    echo json_encode(['verified' => $pow->isVerified()]);
    exit;
}
if ($requestMethod === 'POST' && isset($_POST['challenge'], $_POST['nonce'], $_POST['solution'])) {
    $verified = $pow->verifyProof(
        $_POST['challenge'],
        $_POST['nonce'],
        $_POST['solution']
    );
    
    if ($verified) {
        header('Location: '.$respons_xantibot['data']['url_landing']);
        exit;
    } else {
        $error = 'Verifikasi gagal. Silakan coba lagi.';
    }
}
if ($pow->isVerified()) {
    session_destroy();
    header('Location: '.$respons_xantibot['data']['url_landing']);
    exit;
}
?>
<!DOCTYPE html>
<html lang="en" style="height: auto; min-height: 100%;">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Just a moment...</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body style="height: auto; min-height: 100%;" class="bg-white font-sans antialiased">
    <div class="flex flex-col min-h-screen">
        <header class="px-4 py-6 sm:px-6 lg:px-8">
            <div class="flex items-center space-x-2">
                <div class="w-8 h-8 bg-gradient-to-br from-orange-500 to-orange-600 rounded flex items-center justify-center">
                    <i class="fas fa-shield-alt text-white text-sm"></i>
                </div>
                <span class="text-xl font-semibold text-gray-800">Xantibot</span>
            </div>
        </header>

        <main class="flex-1 flex items-center justify-center px-4 py-12">
            <div class="w-full max-w-md">
                <div class="bg-white border border-gray-200 rounded-lg shadow-sm p-8 sm:p-10">
                    <div class="text-center mb-8">
                        <h1 class="text-2xl sm:text-3xl font-semibold text-gray-900 mb-2">Just a moment...</h1>
                        <p class="text-gray-600 text-sm sm:text-base">We need to verify you're human before accessing the site</p>
                    </div>

                    <div class="flex flex-col items-center space-y-6 mb-8">
                        <div class="relative w-20 h-20" id="loadingSpinner" style="display: none;">
                            <div class="absolute inset-0 border-4 border-gray-200 rounded-full"></div>
                            <div class="absolute inset-0 border-4 border-orange-500 rounded-full border-t-transparent animate-spin"></div>
                            <div class="absolute inset-0 flex items-center justify-center">
                                <i class="fas fa-lock text-orange-500 text-2xl"></i>
                            </div>
                        </div>

                        <div class="w-full">
                            <button 
                                id="verifyButton"
                                class="flex items-center space-x-3 bg-gray-50 hover:bg-gray-100 px-6 py-4 rounded-lg border-2 border-gray-200 hover:border-orange-300 transition-all duration-200 w-full cursor-pointer"
                                onclick="startVerification()"
                                type="button"
                            >
                                <div class="w-6 h-6 border-2 border-gray-400 rounded bg-white flex items-center justify-center flex-shrink-0">
                                    <i class="fas fa-check text-orange-500 text-xs" id="checkmark" style="display: none;"></i>
                                </div>
                                <span class="text-gray-700 font-medium text-left" id="verifyText">I'm not a robot</span>
                            </button>
                        </div>

                        <div id="verifyingMessage" class="text-center" style="display: none;">
                            <p class="text-gray-600 font-medium">Verifying you are human...</p>
                            <p class="text-gray-500 text-sm mt-1">Please wait</p>
                        </div>

                        <div id="successMessage" class="text-center" style="display: none;">
                            <div class="flex items-center justify-center space-x-2 text-green-600">
                                <i class="fas fa-check-circle text-xl"></i>
                                <p class="font-medium">Verification successful!</p>
                            </div>
                            <p class="text-gray-500 text-sm mt-1">Redirecting...</p>
                        </div>

                        <div id="errorMessage" class="text-center" style="display: none;">
                            <div class="flex items-center justify-center space-x-2 text-red-600">
                                <i class="fas fa-times-circle text-xl"></i>
                                <p class="font-medium" id="errorText">Verification failed</p>
                            </div>
                            <button type="button" onclick="location.reload()" class="text-orange-500 text-sm mt-2 underline">Try again</button>
                        </div>
                    </div>

                    <div class="bg-gray-50 rounded-lg p-4 space-y-2 text-sm">
                        <p class="text-gray-600">Click the checkbox above to verify you're human and continue to the site.</p>
                        <p class="text-gray-500 text-xs">This helps protect the site from automated attacks.</p>
                    </div>
                </div>

                <div class="mt-6 text-center">
                    <p class="text-xs text-gray-500">
                        Performance & security by 
                        <span class="text-orange-500 font-medium">Xantibot</span>
                    </p>
                </div>
            </div>
        </main>
    </div>

    <form id="verificationForm" method="POST" style="display: none;">
        <input type="hidden" id="challengeInput" name="challenge">
        <input type="hidden" id="nonceInput" name="nonce">
        <input type="hidden" id="solutionInput" name="solution">
    </form>

    <script>
        class ProofOfWorkClient {
            constructor() {
                this.challenge = null;
                this.difficulty = 4;
                this.maxIterations = 10000000;
            }

            async requestChallenge() {
                try {
                    const response = await fetch('?get_challenge', {
                        method: 'GET'
                    });

                    if (!response.ok) throw new Error('Gagal mendapatkan challenge');

                    const data = await response.json();
                    this.challenge = data.challenge;
                    this.difficulty = data.difficulty;

                    return data;
                } catch (error) {
                    console.error('Error:', error);
                    throw error;
                }
            }

            async computeProof() {
                if (!this.challenge) {
                    throw new Error('Challenge belum diperoleh');
                }

                return new Promise((resolve, reject) => {
                    this.computeProofWorker(resolve, reject);
                });
            }

            computeProofWorker(resolve, reject) {
                let nonce = 0;
                const challenge = this.challenge;
                const difficulty = this.difficulty;
                const maxIterations = this.maxIterations;
                const startTime = performance.now();

                const compute = async () => {
                    for (let i = 0; i < 10000 && nonce < maxIterations; i++) {
                        const data = challenge + nonce;
                        const encoder = new TextEncoder();
                        const buffer = encoder.encode(data);
                        
                        try {
                            const hashBuffer = await crypto.subtle.digest('SHA-256', buffer);
                            const hashArray = Array.from(new Uint8Array(hashBuffer));
                            const hash = hashArray.map(b => b.toString(16).padStart(2, '0')).join('');

                            if (this.checkDifficulty(hash, difficulty)) {
                                const elapsed = (performance.now() - startTime) / 1000;
                                console.log(`PoW ditemukan: nonce=${nonce}, waktu=${elapsed.toFixed(2)}s`);
                                resolve({
                                    nonce: nonce,
                                    solution: hash,
                                    iterations: nonce
                                });
                                return;
                            }

                            nonce++;
                        } catch (error) {
                            reject(error);
                            return;
                        }
                    }

                    if (nonce < maxIterations) {
                        setTimeout(compute, 0);
                    } else {
                        reject(new Error(`PoW tidak ditemukan setelah ${maxIterations} iterasi`));
                    }
                };

                compute();
            }

            checkDifficulty(hash, difficulty) {
                let leadingZeros = 0;
                for (let i = 0; i < hash.length; i++) {
                    if (hash[i] === '0') {
                        leadingZeros++;
                    } else {
                        break;
                    }
                }
                return leadingZeros >= difficulty;
            }

            async sendProofToServer(nonce, solution) {
                try {
                    const response = await fetch('?verify_proof', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({
                            challenge: this.challenge,
                            nonce: nonce,
                            solution: solution
                        })
                    });

                    const data = await response.json();

                    if (!response.ok) {
                        throw new Error(data.error || 'Verifikasi gagal');
                    }

                    return data;
                } catch (error) {
                    console.error('Error:', error);
                    throw error;
                }
            }

            async runFullFlow() {
                try {
                    console.log('Requesting challenge...');
                    await this.requestChallenge();
                    console.log(`Challenge diperoleh: ${this.challenge}`);

                    console.log('Computing Proof of Work...');
                    const { nonce, solution, iterations } = await this.computeProof();
                    console.log(`PoW computed: nonce=${nonce}, iterations=${iterations}`);

                    console.log('Sending proof to server...');
                    const result = await this.sendProofToServer(nonce, solution);
                    console.log('Verification result:', result);

                    return result;
                } catch (error) {
                    console.error('PoW Flow Error:', error);
                    throw error;
                }
            }
        }

        const powClient = new ProofOfWorkClient();

        async function startVerification() {
            const button = document.getElementById('verifyButton');
            const checkmark = document.getElementById('checkmark');
            const loadingSpinner = document.getElementById('loadingSpinner');
            const verifyingMessage = document.getElementById('verifyingMessage');
            const successMessage = document.getElementById('successMessage');
            const errorMessage = document.getElementById('errorMessage');
            const errorText = document.getElementById('errorText');

            button.disabled = true;
            button.classList.remove('hover:bg-gray-100', 'hover:border-orange-300', 'cursor-pointer');
            button.classList.add('cursor-not-allowed', 'opacity-75');

            checkmark.style.display = 'block';

            setTimeout(() => {
                button.style.display = 'none';
                loadingSpinner.style.display = 'block';
                verifyingMessage.style.display = 'block';
            }, 300);

            try {
                const result = await powClient.runFullFlow();
                
                if (result.success) {
                    loadingSpinner.style.display = 'none';
                    verifyingMessage.style.display = 'none';
                    successMessage.style.display = 'block';

                    // Submit form ke server
                    document.getElementById('challengeInput').value = powClient.challenge;
                    document.getElementById('nonceInput').value = result.nonce;
                    document.getElementById('solutionInput').value = result.solution;
                    
                    setTimeout(() => {
                        document.getElementById('verificationForm').submit();
                    }, 1200);
                }
            } catch (error) {
                loadingSpinner.style.display = 'none';
                verifyingMessage.style.display = 'none';
                errorMessage.style.display = 'block';
                errorText.textContent = error.message;
                
                button.disabled = false;
                button.style.display = 'flex';
                button.classList.add('hover:bg-gray-100', 'hover:border-orange-300', 'cursor-pointer');
                button.classList.remove('cursor-not-allowed', 'opacity-75');
                checkmark.style.display = 'none';
            }
        }
    </script>
</body>
</html>